<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\AppData;
use App\Models\Otp;
use App\Models\User;
use App\Notifications\BonusWonNotification;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use PgSql\Lob;

class AuthController extends Controller
{

    public function login(Request $request)
    {
        $request->validate([
            'phone' => 'required|regex:/[0-9]{10}/|digits:10',
            'mpin' => 'required|string',
            'fcm' => 'sometimes|string',
        ]);

        $user = User::where('phone', $request->phone)
            ->with('withdrawDetails')
            ->first();
        if ($user === NULL) {
            $message = "User Not Found! Please Create Account First!";
            return response()->failed($message);
        }

        if (!$user || !Hash::check($request->mpin, $user->password)) {
            $message = "Invalid Credentials!";
            return response()->failed($message);
        }
        if ($user->role == "admin" || $user->role == "sub-admin") {
            $message = "You are not user";
            return response()->failed($message);
        }


        //if we check this then user can't login
        // if (!$user->confirmed) {
        //     $message = "Please Verify Your Account!";
        //     return response()->failed($message);
        // }

        if ($user->blocked) {
            $message = "Your Account Has Been Blocked!";
            return response()->failed($message);
        }

        if (isset($request->fcm)) {
            $user->update(array('fcm' => $request->fcm));
        }

        $token = $user->createToken('auth-token')->plainTextToken;
        return response()->success("Login Done!", compact('token', 'user'));
    }

    public function create(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'phone' => 'required|unique:users,phone|regex:/[0-9]{10}/|digits:10',
            'password' => 'required|string',
            'refferal_code' => 'nullable|numeric|sometimes'
        ]);

        $user_check = User::where('phone', $request->phone)->first();
        if ($user_check !== NULL) {
            $message = "User Already Exist!!";
            return response()->failed($message);
        }

        if (!is_null($request->refferal_code)) {
            $refferalUser = User::where('own_code', $request->refferal_code)->first();
            if (is_null($refferalUser)) {
                $message = "Refferal code is Invalid";
                return response()->failed($message);
            }
        }
        $appData = AppData::first();

        $user =  User::create([
            'name' => $request->name,
            'phone' => $request->phone,
            'balance' => $appData->welcome_bonus,
            'password' => Hash::make($request->password),
            'role' => 'user',
            "user_id" => isset($refferalUser) ? $refferalUser->id : NULL,
            'own_code' => rand(1000000, 9999999)
        ]);
        if (isset($refferalUser)) {
            $user->notify(new BonusWonNotification($appData->invite_bonus, $user->fcm));
        }
        return response()->success("Done", compact('user'));
    }
    
    public function confirm(Request $request)
    {
        $request->validate([
            'otp' => 'required|string',
            'phone' => 'required|numeric'
        ]);
        $otp = $request->otp;
        $phone = $request->phone;
        if (!OtpController::verifyMeraOtp($otp, $phone)) {
            $message = "Otp is Invalid";
            return response()->failed($message);
        }
        $user = User::where('phone', $phone)->first();
        $user->confirmed = 1;
        $user->save();

        $token = $user->createToken('auth-token')->plainTextToken;
        $message = "Account Confirmed Successfully!";
        return response()->success($message, compact('user', 'token'));
    }

    // ------- OLD CODE
    // public function confirm(Request $request)
    // {
    //     // Log::info("confirm request", "otp is " . $request->otp);
    //     $request->validate([
    //         'otp' => 'required|string'
    //     ]);
    //     $otp = Otp::where('otp', $request->otp)
    //         ->with('user')
    //         ->latest()->first();
    //     $user =  $otp->user;
    //     if (is_null($otp)) {
    //         $message = "Otp is Invalid";
    //         return response()->failed($message);
    //     }
    //     $otp_created_at = Carbon::parse($otp->created_at);
    //     $time_diff = $otp_created_at->diffInSeconds(Carbon::now());
    //     if ($time_diff > 300) {
    //         $message = "Otp Expired!";
    //         return response()->failed($message);
    //     } else {
    //         $user->confirmed = 1;
    //         $user->save();

    //         $token = $user->createToken('auth-token')->plainTextToken;
    //         $message = "Account Confirmed Successfully!";
    //         return response()->success($message, compact('user', 'token'));
    //     }
    // }

    public function forgetPasswordOtp(Request $request)
    {
        $request->validate([
            'phone' => 'required|numeric'
        ]);

        $user = User::where('phone', $request->phone)->first();
        if (blank($user)) {
            $message = "User not found";
            return response()->failed($message);
        }
        $otpController = new OtpController();
        $otp = rand(1000, 9999);
        $user->otps()->create([
            'otp' => $otp,
            'created_at' => now(),
        ]);
        $phoneNumber = $user->phone;
        // $message = "Your forget password OTP is " . $otp;
        $message = $otp;
        $otpController->sendOtpSms($message, $phoneNumber);
        return response()->success($message, compact('otp'));
    }
    
    public function forgetPasswordVerify(Request $request)
    {
        $request->validate([
            'phone' => 'required|numeric',
            'mpin' => 'required|numeric',
            'otp' => 'required|string'
        ]);

        $user = User::where('phone', $request->phone)->first();
        if (is_null($user)) {
            $message = "User not found";
            return response()->failed($message);
        }


        $otpResponse = OtpController::verifyMeraOtp($request->otp, $request->phone);
        if ($otpResponse == false) {
            $message = "Otp is Invalid";
            return response()->failed($message);
        }

        //if password starts with 0, then return error
        if (substr($request->mpin, 0, 1) == 0) {
            $message = "Password can't start with 0";
            return response()->failed($message);
        }
        $user->password = Hash::make($request->mpin);
        $user->save();
        return response()->success("Password Has Been Updated!", compact('user'));
    }

    // ------- OLD CODE
    // public function forgetPasswordVerify(Request $request)
    // {
    //     $request->validate([
    //         'phone' => 'required|numeric',
    //         'mpin' => 'required|numeric',
    //         'otp' => 'required|string'
    //     ]);

    //     $user = User::where('phone', $request->phone)->first();
    //     if (is_null($user)) {
    //         $message = "User not found";
    //         return response()->failed($message);
    //     }
    //     $otp = Otp::where('otp', $request->otp)->where('user_id', $user->id)->latest()->first();
    //     if (is_null($otp)) {
    //         $message = "Otp is Invalid";
    //         return response()->failed($message);
    //     }
    //     $otp_created_at = Carbon::parse($otp->created_at);
    //     $time_diff = $otp_created_at->diffInSeconds(Carbon::now());
    //     if ($time_diff > 180) {
    //         $message = "Otp Expired!";
    //         return response()->failed($message);
    //     } else {
    //         $otp->delete();
    //         $user->password = Hash::make($request->mpin);
    //         $user->save();
    //         return response()->success("Password Has Been Updated!", compact('user'));
    //     }
    // }


    public function logout(Request $request)
    {
        /** @var User $user  */
        $user =  Auth::user();
        $user->fcm = NULL;
        $user->save();
        $request->user()->currentaccesstoken()->delete();
        $message = "Logout Successfully";
        return response()->success($message, NULL);
    }
}
