<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\DesawarMarket;
use App\Models\DesawarRecord;
use App\Models\DesawarResult;
use App\Models\GameType;
use App\Models\Market;
use App\Models\MarketRecord;
use App\Models\MarketResult;
use App\Models\StartLineMarket;
use App\Models\StartLineRecord;
use App\Models\StartLineResult;
use Carbon\Carbon;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Illuminate\Support\Str;

class GameController extends Controller
{
    public function submitGame(Request $request)
    {
        $request->validate([
            'type' => ['required', Rule::in(['general', 'startLine', 'desawar'])],
            'market_id' => 'required|numeric',
            'games' => [
                'required',
                'array',
                'min:1',
                function (string $attribute, mixed $value, Closure $fail) {
                    foreach ($value as $game) {
                        if (!is_array($game) || !isset($game['number']) || !isset($game['amount']) || !isset($game['session']) || !isset($game['game_type_id'])) {
                            $fail("{$attribute} must be an array of games with number, amount, game_type_id and session.");
                        }
                    }
                },
            ],
            'games.*.number' => ['required', 'string'],
            'games.*.amount' => ['required', 'numeric'],
            'games.*.session' => ["required", Rule::in(['open', 'close', 'null'])],
            'games.*.game_type_id' => 'required|exists:game_types,id',
        ]);
        $market = match ($request->type) {
            "general" => Market::class,
            "startLine" => StartLineMarket::class,
            "desawar" => DesawarMarket::class,
        };
        if (!$market::where('id', $request->market_id)->exists()) {
            $message = "The selected market id is invalid.";
            return response()->failed($message);
        }

        if (!$market::where('id', $request->market_id)->first()->game_on) {
            return response()->failed("Game Is Not Open at The Moment!");
        }

        $amountSum = array_sum(array_column($request->games, 'amount'));
        /** @var User $user  */
        $user = auth()->user();
        if ($user->balance < $amountSum) {
            return response()->failed("User's balance is insufficient");
        }

        $marketRecord = match ($request->type) {
            "general" => MarketRecord::class,
            "startLine" => StartLineRecord::class,
            "desawar" => DesawarRecord::class,
        };
        $property_name = match ($request->type) {
            "general" => "market_id",
            "startLine" => "startline_market_id",
            "desawar" => "desawar_market_id",
        };

        foreach ($request->games as $game) {

            $market = $market::find($request->market_id);

            //check if game type is open, if open then check open_game_status
            if (($game['session'] == 'open' || $game['game_type_id'] == 2 || $game['game_type_id'] == 6 || $game['game_type_id'] == 7 || $game['game_type_id'] == 8) && $request->type != 'startLine') {
                if (!$market->open_game_status) {
                    return response()->failed("Open Game is not open at the moment!");
                }
            }

            //check if game type is close, if close then check close_game_status
            if ($game['session'] == 'close') {
                if (!$market->close_game_status) {
                    return response()->failed("Close Game is not open at the moment!");
                }
            }

            $marketRecord::create([
                "market_id" => $request->market_id,
                'game_type_id' => $game['game_type_id'],
                $property_name => $request->market_id,
                'number' => $game['number'],
                'amount' => $game['amount'],
                'session' => $game['session'],
                'status' => "pending",
                'user_id' => $user->id,
                'game_string' => Str::random(12),
                'date' => Carbon::today()
            ]);

            $game_type = GameType::find($request->game_type_id);

            $user->transactions()->create([
                "previous_amount" => $user->balance,
                "amount" => $game["amount"],
                "current_amount" => $user->balance - $game["amount"],
                "type" => "play",
                "details" => "Game ( " . Carbon::now() . " : $market->name : $game_type->name : $game[session] ) : $game[number]",
            ]);

            $user->balance -= $game["amount"];
            $user->save();
            $user->refresh();
        }

        $balance_left = $user->balance;
        $message = "Games successfully added";
        return response()->success($message, compact('balance_left'));
    }

    public function getGameHistory(Request $request)
    {
        $request->validate([
            'type' => ['required', Rule::in(['general', 'startLine', 'desawar'])],
            'page' => 'required|numeric'
        ]);
        $market = match ($request->input('type')) {
            "general" => MarketRecord::class,
            "startLine" => StartLineRecord::class,
            "desawar" => DesawarRecord::class,
        };
        $user = Auth::user();
        $gameHistory = $market::with(['market', 'gameType'])
            ->where('user_id', $user->id)->latest()->paginate(10, ['*'], 'game_history', $request->page);
        return response()->success("Data Sent", compact('gameHistory'));
    }

    public function getGameResults(Request $request)
    {
        $request->validate([
            'type' => ['required', Rule::in(['general', 'startLine', 'desawar'])],
            'date' => 'date|nullable|sometimes'
        ]);
        $market = match ($request->input('type')) {
            "general" => MarketResult::class,
            "startLine" => StartLineResult::class,
            "desawar" => DesawarResult::class,
        };

        $date = $request->input('date') ?? Carbon::today();
        $gameResults = $market::with('market')
            ->where('result_date', $date)
            ->get();
        return response()->success("Data Sent!", compact('gameResults'));
    }

    public function getGameRates()
    {
        $gameTypesGeneral = GameType::where('type', 'general')->get();
        $gameTypesStartLine = GameType::where('type', 'start_line')->get();
        $gameTypesDesawar = GameType::where('type', 'desawar')->get();

        $data = [
            $data1 = [
                'title' => 'Game Win Ratio for All Bids',
                'list' => $gameTypesGeneral
            ],
            $data2 = [
                'title' => 'King Starline Game Win Ration',
                'list' => $gameTypesStartLine
            ],
            $data3 = [
                'title' => 'Desawar Game Win Ratio',
                'list' => $gameTypesDesawar
            ],
        ];

        return response()->success("Data Sent!", compact('data'));
    }
}
