<?php

namespace App\Http\Controllers\Dashboard;

use App\Models\Market;
use App\Models\MarketRecord;
use App\Http\Controllers\Controller;
use App\Models\GameType;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

use function GuzzleHttp\Promise\all;

class MarketRecordController extends Controller
{
    public function index(Request $request)
    {
        if ($request->has('searchValue')) {
            $searchValue = $request->searchValue;
            $marketRecords = MarketRecord::with('market')
                ->whereHas('user', function ($query) use ($searchValue) {
                    $query->where('name', 'LIKE', '%' . $searchValue . '%')
                        ->orWhere('phone', 'LIKE', '%' . $searchValue . '%');
                })->latest()->paginate(250);
            return view('dashboard.markets.records', compact('marketRecords', 'searchValue'));
        }
        $marketRecords = MarketRecord::with('market')->latest()->paginate(25);
        return view('dashboard.markets.records', compact('marketRecords'));
    }

    public function winHistory(Request $request)
    {
        if ($request->has('searchValue')) {
            $searchValue = $request->searchValue;
            $winHistory = MarketRecord::with(['user', 'market', 'gameType'])
                ->where('status', 'success')
                ->whereHas('user', function ($query) use ($searchValue) {
                    $query->where('name', 'LIKE', '%' . $searchValue . '%')
                        ->orWhere('phone', 'LIKE', '%' . $searchValue . '%');
                })->latest()->paginate(250);
            return view('dashboard.markets.win-history', compact('winHistory', 'searchValue'));
        }
        $winHistory = MarketRecord::with(['user', 'market', 'gameType'])
            ->where('status', 'success')->latest()->paginate(25);
        return view('dashboard.markets.win-history', compact('winHistory'));
    }

    public function data(Request $request)
    {
        $records = GameType::where('type', 'general')->orderBy('id', 'ASC')->get();
        $data = [];
        $count = 0;
        $today = Carbon::today();
        foreach ($records as $record) {

            if ($request->market_id && $request->market_time) {
                $bids = MarketRecord::where([
                    'game_type_id' => $record->id,
                    'market_id' => $request->market_id,
                    'session' => $request->market_time
                ])
                    ->whereDate('created_at', $today)
                    ->select('game_type_id', 'number', DB::raw('SUM(amount) as total_amount'))
                    ->groupBy('number')
                    ->orderBY('number', 'ASC')
                    ->get();
            } else {
                $bids = MarketRecord::where('game_type_id', $record->id)
                    ->whereDate('created_at', $today)
                    ->select('game_type_id', 'number', DB::raw('SUM(amount) as total_amount'))
                    ->groupBy('number')
                    ->orderBY('number', 'ASC')
                    ->get();
            }
            if (count($bids) > $count) {
                $count = count($bids);
            }
            $bids->gameType = $record;
            array_push($data, $bids);
        }
        $markets = Market::all();

        return view('dashboard.markets.data', compact('data', 'count', 'markets'));
    }
    
    
    //  Revert Records 
    public function reverts(Request $request){
        if($request->has('searchValue')) {
            $searchValue = $request->searchValue;
            $marketReverts = MarketRecord::with('market')
                ->whereHas('user', function ($query) use ($searchValue) {
                    $query->where('name', 'LIKE', '%'. $searchValue . '%')
                            ->orWhere('phone', 'LIKE', '%'. $searchValue. '%');
            })->where('status', 'pending')->latest()->paginate(250);
            return view('dashboard.markets.reverts', compact('marketReverts', 'searchValue'));
        } 
        
        $marketReverts = MarketRecord::with('market')->where('status', 'pending')->latest()->paginate(250);
        return view('dashboard.markets.reverts', compact('marketReverts'));
    }

    public function revertRecord($id){
        $marketRecord = MarketRecord::with('user')->findOrFail($id);
        $user = $marketRecord->user;

        abort_if(! $marketRecord->status === 'PENDING', 404);

        $marketRecord->status = 'revert';
        $user->balance += $marketRecord->amount;
        $user->save();
        $marketRecord->save();

        $user->transactions()->create([
            'previous_amount' =>  $user->balance - $marketRecord->amount,
            'amount' => $marketRecord->amount,
            'current_amount' => $user->balance,
            "type" => "play",
            "details" => "bet revert"
        ]);

        return redirect()->back()->with('success', "Bet revert Successfully.");
    }
}
