<?php

namespace App\Services;

use App\Models\DesawarMarket;
use App\Models\DesawarRecord;
use App\Models\User;
use App\Notifications\GameResultNotification;
use App\Notifications\GameWinNotification;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;

class DesawarResultSetService
{

    public function setResult($request, $fromApi = false)
    {
        $first_digit = str_repeat(strval($request->digit)[0], 3);
        $second_digit = str_repeat(strval($request->digit)[1], 3);
        $digit = strval($request->digit);

        $market = DesawarMarket::with(['results' => function ($query) use ($request) {
            $query->where('result_date', $request->date)->first();
        }])->where('id', $request->market)
            ->firstOrFail();
        $result = $market->results->first();
        if (isset($result)) {
            return back()->with("error", "Result is already declared.");
        }
        $market->results()->create([
            "result_date" => $request->date,
            "result" => $request->digit,
            "first_digit_of_result" => $first_digit,
            "second_digit_of_result" => $second_digit
        ]);

        if ($market->previous_day_check && $fromApi) {
            $date = Carbon::parse($request->date);
            $date = $date->subDay();
        } else {
            $date = $request->date;
        }

        $this->checkUsersWonOrLost($market->id, $first_digit, $second_digit, $digit, $date);
        return back()->with("success", "Market result created successfully");
    }

    private function checkUsersWonOrLost($marketId, $first_digit, $second_digit, $digit, $date)
    {
        Log::info("first digit: " . $first_digit . " second digit: " . $second_digit . " digit: " . $digit . " date: " . $date);
        $market_name = DesawarMarket::find($marketId)->name;
        $win_numbers = [$first_digit, $second_digit, $digit];
        $records = DesawarRecord::with(["market", "gameType", "user"])
            ->where('date', $date)
            ->where("desawar_market_id", $marketId)
            ->where('status', 'pending')
            ->get();
        if (filled($records)) {
            foreach ($records as $record) {
                $user = $record->user;
                if (in_array($record->number, $win_numbers, true)) {
                    //log record number and win numbrs
                    // Log::info("Record number: " . $record->number . " Win numbers: " . json_encode($win_numbers));
                    $record["status"] = "success";
                    $record["win_amount"] = $record->amount * $record->gameType->multiply_by;
                    $user->balance += $record["win_amount"];
                    $user->save();
                    $user->transactions()->create([
                        'previous_amount' =>  $user->balance - $record["win_amount"],
                        'amount' => $record["win_amount"],
                        'current_amount' => $user->balance,
                        "type" => "win",
                        // ( current_time : game_name : game_type : session ) : number
                        "details" => "Win (" . now() . $record->market->name .
                            " :" . $record->gameType->name . ": " . " ) : $record->number"
                    ]);
                    if ($user->desawar_noti) {
                        $user->notify(new GameWinNotification($record['win_amount'], $user->fcm));
                    }
                } else {
                    $record["status"] = "failed";
                    $record["win_amount"] = 0;
                }
                $record->save();
            }
        }
        $usersFcms = User::where('desawar_noti', true)->pluck('fcm')->toArray();
        Notification::send(NULL, new GameResultNotification($market_name, $digit, $usersFcms));
    }
}
